<?php

/**
 * Realgest Admin Class
 * 
 * Handles admin settings and functionality
 */
class Realgest_Admin {

    /**
     * API instance
     */
    private $api = null;

    /**
     * Constructor
     */
    public function __construct($api) {
        $this->api = $api;

        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Register settings
        add_action('admin_init', array($this, 'register_settings'));
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Impostazioni Realgest', 'realgest-for-wordpress'),
            __('Realgest', 'realgest-for-wordpress'),
            'manage_options',
            'realgest-settings',
            array($this, 'settings_page'),
            'dashicons-admin-home',
            30
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        // Register settings
        register_setting('realgest_settings', 'realgest_api_key');
        register_setting('realgest_settings', 'realgest_agency_id');
        register_setting('realgest_settings', 'realgest_google_maps_api_key');

        // Add settings section
        add_settings_section(
            'realgest_settings_section',
            __('Impostazioni API', 'realgest-for-wordpress'),
            array($this, 'settings_section_callback'),
            'realgest_settings'
        );

        // Add settings fields
        add_settings_field(
            'realgest_api_key',
            __('Realgest API Key', 'realgest-for-wordpress'),
            array($this, 'api_key_callback'),
            'realgest_settings',
            'realgest_settings_section'
        );

        add_settings_field(
            'realgest_agency_id',
            __('ID Agenzia', 'realgest-for-wordpress'),
            array($this, 'agency_id_callback'),
            'realgest_settings',
            'realgest_settings_section'
        );

        add_settings_field(
            'realgest_google_maps_api_key',
            __('Google Maps API Key', 'realgest-for-wordpress'),
            array($this, 'google_maps_api_key_callback'),
            'realgest_settings',
            'realgest_settings_section'
        );
    }

    /**
     * Settings section callback
     */
    public function settings_section_callback() {
        echo '<p>' . __('Inserisci le tue credenziali Realgest API qui sotto.', 'realgest-for-wordpress') . '</p>';
        echo '<p>' . __('Per ricavare "Realgest API Key" e "ID Agenzia" accedi alla dashboard admin del tuo account Realgest e le troverai nella sezione  ', 'realgest-for-wordpress') . '<a target="_blank" href="https://software.realgest.it/admin/a/gestione/api">API</a></p>';
    }

    /**
     * API key callback
     */
    public function api_key_callback() {
        $api_key = get_option('realgest_api_key');
        echo '<input type="text" name="realgest_api_key" value="' . esc_attr($api_key) . '" class="regular-text">';

    }

    /**
     * Agency ID callback
     */
    public function agency_id_callback() {
        $agency_id = get_option('realgest_agency_id');
        echo '<input type="text" name="realgest_agency_id" value="' . esc_attr($agency_id) . '" class="regular-text">';
    }

    /**
     * Google Maps API key callback
     */
    public function google_maps_api_key_callback() {
        $google_maps_api_key = get_option('realgest_google_maps_api_key');
        echo '<input type="text" name="realgest_google_maps_api_key" value="' . esc_attr($google_maps_api_key) . '" class="regular-text">';
        echo '<p class="description">' . __('Inserisci la tua chiave API di Google Maps per abilitare le mappe nelle pagine dei dettagli delle proprietà.', 'realgest-for-wordpress') . '</p>';
    }

    /**
     * Settings page
     */
    public function settings_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <form method="post" action="options.php">
                <?php
                settings_fields('realgest_settings');
                do_settings_sections('realgest_settings');
                submit_button();
                ?>
            </form>

            <?php if ($this->api->is_configured()): ?>
                <div class="notice notice-success">
                    <p><?php _e('La connessione API è configurata correttamente.', 'realgest-for-wordpress'); ?></p>
                </div>

                <h2><?php _e('Shortcodes', 'realgest-for-wordpress'); ?></h2>

                <div class="realgest-shortcodes">
                    <div class="realgest-shortcode">
                        <h3><?php _e('Elenco Immobili', 'realgest-for-wordpress'); ?></h3>
                        <p><?php _e('Usa questo shortcode per mostrate l\'elenco degli immobili:', 'realgest-for-wordpress'); ?></p>
                        <code>[realgest_properties limit="10" template="grid" pagination="true"]</code>

                        <h4><?php _e('Attributi disponibili', 'realgest-for-wordpress'); ?></h4>
                        <ul>
                            <li><code>limit</code>
                                - <?php _e( 'Numero di immobili da visualizzare per pagina (default: 10)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>type</code> - <?php _e( 'ID tipo immobile', 'realgest-for-wordpress' ); ?></li>
                            <li><code>offer</code>
                                - <?php _e( 'ID tipo offerta (vendita/affitto)', 'realgest-for-wordpress' ); ?></li>
                            <li><code>class</code>
                                - <?php _e( 'ID classe immobile (residenziale/commerciale)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>city</code> - <?php _e( 'ID città', 'realgest-for-wordpress' ); ?></li>
                            <li><code>zone</code> - <?php _e( 'ID zona', 'realgest-for-wordpress' ); ?></li>
                            <li><code>price_min</code> - <?php _e( 'Prezzo minimo', 'realgest-for-wordpress' ); ?></li>
                            <li><code>price_max</code> - <?php _e( 'Prezzo massimo', 'realgest-for-wordpress' ); ?></li>
                            <li><code>surface_min</code> - <?php _e( 'Superficie minima', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>surface_max</code>
                                - <?php _e( 'Superficie massima', 'realgest-for-wordpress' ); ?></li>
                            <li><code>rooms_min</code>
                                - <?php _e( 'Numero minimo di locali', 'realgest-for-wordpress' ); ?></li>
                            <li><code>rooms_max</code>
                                - <?php _e( 'Numero massimo di locali', 'realgest-for-wordpress' ); ?></li>
                            <li><code>bathrooms_min</code>
                                - <?php _e( 'Numero minimo di bagni', 'realgest-for-wordpress' ); ?></li>
                            <li><code>bathrooms_max</code>
                                - <?php _e( 'Numero massimo di bagni', 'realgest-for-wordpress' ); ?></li>
                            <li><code>garden</code>
                                - <?php _e( 'Ha giardino (vero/falso)', 'realgest-for-wordpress' ); ?></li>
                            <li><code>terrace</code>
                                - <?php _e( 'Ha terrazzo (vero/falso)', 'realgest-for-wordpress' ); ?></li>
                            <li><code>balcony</code>
                                - <?php _e( 'Ha balcone (vero/falso)', 'realgest-for-wordpress' ); ?></li>
                            <li><code>elevator</code>
                                - <?php _e( 'Ha ascensore (vero/falso)', 'realgest-for-wordpress' ); ?></li>
                            <li><code>cellar</code>
                                - <?php _e( 'Ha cantina (vero/falso)', 'realgest-for-wordpress' ); ?></li>
                            <li><code>air_conditioning</code>
                                - <?php _e( 'Ha aria condizionata (vero/falso)', 'realgest-for-wordpress' ); ?></li>
                            <li><code>furnished</code>
                                - <?php _e( 'È arredato (vero/falso)', 'realgest-for-wordpress' ); ?></li>
                            <li><code>only_with_photos</code>
                                - <?php _e( 'Mostra solo immobili con foto (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>orderby</code>
                                - <?php _e( 'Ordina per campo (default: data_inserimento)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>order</code>
                                - <?php _e( 'Direzione ordinamento (ASC/DESC, default: DESC)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>template</code>
                                - <?php _e( 'Template to use: grid/vertical (default: grid)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>columns</code>
                                - <?php _e( 'Numero di colonne per il template grid (default: 3)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>pagination</code>
                                - <?php _e( 'Mostra paginazione (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                        </ul>
                    </div>

                    <div class="realgest-shortcode">
                        <h3><?php _e( 'Singolo Immobile', 'realgest-for-wordpress' ); ?></h3>
                        <p><?php _e( 'Usa questo shortcode per mostrare un singolo immobile:', 'realgest-for-wordpress' ); ?></p>
                        <code>[realgest_property id="123" template="default"]</code>

                        <h4><?php _e('Attributi disponibili', 'realgest-for-wordpress'); ?></h4>
                        <ul>
                            <li><code>id</code> - <?php _e('ID Immobile', 'realgest-for-wordpress'); ?></li>
                            <li><code>slug</code> - <?php _e('Slug Immobile (alternativo all\'ID)', 'realgest-for-wordpress'); ?></li>
                            <li><code>template</code> - <?php _e('Template da usare (default: default)', 'realgest-for-wordpress'); ?></li>
                        </ul>
                    </div>

                    <div class="realgest-shortcode">
                        <h3><?php _e('Ricerca Immobili', 'realgest-for-wordpress'); ?></h3>
                        <p><?php _e('Usa questo shortcode per mostrare un form di ricerca immobili:', 'realgest-for-wordpress'); ?></p>
                        <code>[realgest_search template="default" action=""]</code>

                        <h4><?php _e('Attributi disponibili', 'realgest-for-wordpress'); ?></h4>
                        <ul>
                            <li><code>template</code>
                                - <?php _e( 'Template da usare (default: default)', 'realgest-for-wordpress' ); ?></li>
                            <li><code>action</code>
                                - <?php _e( 'URL di destinazione del form (default: pagina corrente)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>show_offer_types</code>
                                - <?php _e( 'Mostra tipi di offerta (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>show_property_types</code>
                                - <?php _e( 'Mostra tipi di immobile (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>show_cities</code>
                                - <?php _e( 'Mostra città (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>show_zones</code>
                                - <?php _e( 'Mostra zone (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>show_price</code>
                                - <?php _e( 'Mostra intervallo di prezzo (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>show_surface</code>
                                - <?php _e( 'Mostra intervallo di superficie (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>show_rooms</code>
                                - <?php _e( 'Mostra intervallo locali (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>show_bathrooms</code>
                                - <?php _e( 'Mostra intervallo bagni (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                            <li><code>show_features</code>
                                - <?php _e( 'Mostra caratteristiche (vero/falso, default: vero)', 'realgest-for-wordpress' ); ?>
                            </li>
                        </ul>
                    </div>
                </div>
            <?php else: ?>
                <div class="notice notice-warning">
                    <p><?php _e( 'Configura le tue credenziali API per utilizzare il plugin.', 'realgest-for-wordpress' ); ?></p>
                </div>
            <?php endif; ?>
        </div>

        <style>
            .realgest-shortcodes {
                margin-top: 20px;
            }

            .realgest-shortcode {
                background: #fff;
                border: 1px solid #ccd0d4;
                padding: 15px;
                margin-bottom: 20px;
            }

            .realgest-shortcode h3 {
                margin-top: 0;
            }

            .realgest-shortcode code {
                display: block;
                padding: 10px;
                margin: 10px 0;
                background: #f5f5f5;
            }

            .realgest-shortcode ul {
                list-style-type: disc;
                padding-left: 20px;
            }
        </style>
        <?php
    }
}
