/**
 * Realgest for WordPress JavaScript
 */

(function($) {
    'use strict';

    // Initialize when document is ready
    $(document).ready(function() {
        // Initialize property gallery
        initPropertyGallery();

        // Initialize search form
        initSearchForm();

        // Initialize Google Maps
        initGoogleMaps();
    });

    /**
     * Initialize property gallery
     */
    function initPropertyGallery() {
        // Check if we're on a property detail page
        if ($('.realgest-property-gallery').length) {
            // Handle thumbnail click
            $('.realgest-property-gallery-thumbnail').on('click', function() {
                var thumbnailImg = $(this).find('img');
                var mainImg = $('.realgest-property-gallery-main-image');

                // Update main image
                mainImg.attr('src', thumbnailImg.attr('src'));
                mainImg.attr('alt', thumbnailImg.attr('alt'));

                // Add active class to clicked thumbnail
                $('.realgest-property-gallery-thumbnail').removeClass('active');
                $(this).addClass('active');
            });

            // Add active class to first thumbnail
            $('.realgest-property-gallery-thumbnail:first').addClass('active');
        }
    }

    /**
     * Initialize search form
     */
    function initSearchForm() {
        // Handle city change to load zones
        $('.realgest-city-select').on('change', function() {
            var cityId = $(this).val();
            var zoneSelect = $('.realgest-zone-select');

            if (cityId) {
                // Enable zone select
                zoneSelect.prop('disabled', false);

                // Clear current options
                zoneSelect.find('option:not(:first)').remove();

                // Show loading
                zoneSelect.after('<span class="realgest-loading">' + realgest_params.loading_text + '</span>');

                // Load zones via AJAX
                $.ajax({
                    url: realgest_params.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'realgest_get_zones',
                        city_id: cityId,
                        nonce: realgest_params.nonce
                    },
                    success: function(response) {
                        // Remove loading
                        $('.realgest-loading').remove();

                        if (response.success && response.data.zones) {
                            // Add new options
                            $.each(response.data.zones, function(index, zone) {
                                zoneSelect.append($('<option>', {
                                    value: zone.id,
                                    text: zone.name
                                }));
                            });
                        }
                    },
                    error: function() {
                        // Remove loading
                        $('.realgest-loading').remove();
                    }
                });
            } else {
                // Disable zone select
                zoneSelect.prop('disabled', true);
                zoneSelect.find('option:not(:first)').remove();
            }
        });

        // Handle price range inputs
        $('#realgest-price-min, #realgest-price-max').on('input', function() {
            validateRangeInputs('realgest-price-min', 'realgest-price-max');
        });

        // Handle surface range inputs
        $('#realgest-surface-min, #realgest-surface-max').on('input', function() {
            validateRangeInputs('realgest-surface-min', 'realgest-surface-max');
        });

        // Handle rooms range inputs
        $('#realgest-rooms-min, #realgest-rooms-max').on('input', function() {
            validateRangeInputs('realgest-rooms-min', 'realgest-rooms-max');
        });

        // Handle bathrooms range inputs
        $('#realgest-bathrooms-min, #realgest-bathrooms-max').on('input', function() {
            validateRangeInputs('realgest-bathrooms-min', 'realgest-bathrooms-max');
        });
    }

    /**
     * Validate range inputs
     * 
     * @param {string} minId ID of minimum input
     * @param {string} maxId ID of maximum input
     */
    function validateRangeInputs(minId, maxId) {
        var minInput = $('#' + minId);
        var maxInput = $('#' + maxId);

        if (minInput.val() && maxInput.val()) {
            var minValue = parseInt(minInput.val());
            var maxValue = parseInt(maxInput.val());

            if (minValue > maxValue) {
                maxInput.val(minValue);
            }
        }
    }

    /**
     * Initialize Google Maps
     */
    function initGoogleMaps() {
        // Check if we're on a property detail page with a map
        var mapContainer = $('#realgest-property-map');
        if (mapContainer.length) {
            // Get coordinates from data attributes
            var lat = parseFloat(mapContainer.data('lat'));
            var lng = parseFloat(mapContainer.data('lng'));

            // Check if Google Maps API is loaded
            if (typeof google !== 'undefined' && typeof google.maps !== 'undefined') {
                // Create map
                var map = new google.maps.Map(mapContainer[0], {
                    center: {lat: lat, lng: lng},
                    zoom: 15
                });

                // Add marker
                var marker = new google.maps.Marker({
                    position: {lat: lat, lng: lng},
                    map: map
                });
            } else {
                // If Google Maps API is not loaded yet, wait for it
                window.initRealgestMap = function() {
                    // Create map
                    var map = new google.maps.Map(mapContainer[0], {
                        center: {lat: lat, lng: lng},
                        zoom: 15
                    });

                    // Add marker
                    var marker = new google.maps.Marker({
                        position: {lat: lat, lng: lng},
                        map: map
                    });
                };
            }
        }
    }



})(jQuery);
// ===== Realgest: gallery with background-image navigation =====
(function(){
    function initGallery(root){
        var main     = root.querySelector('[data-rg-main-img]');
        var thumbs   = Array.prototype.slice.call(root.querySelectorAll('[data-rg-thumb]'));
        if(!main || !thumbs.length) return;

        var prevBtn  = root.querySelector('[data-rg-prev]');
        var nextBtn  = root.querySelector('[data-rg-next]');

        var sources = thumbs.map(function(t){ return t.style.backgroundImage; }).filter(Boolean);
        var current = 0;

        function setActive(i){
            current = (i + sources.length) % sources.length;
            main.style.backgroundImage = sources[current];
            thumbs.forEach(function(t){ t.classList.remove('is-active'); });
            var active = thumbs[current];
            if(active) active.classList.add('is-active');
        }

        thumbs.forEach(function(t, i){
            t.addEventListener('click', function(){ setActive(i); });
        });

        if(prevBtn) prevBtn.addEventListener('click', function(){ setActive(current - 1); });
        if(nextBtn) nextBtn.addEventListener('click', function(){ setActive(current + 1); });

        root.setAttribute('tabindex', '0');
        root.addEventListener('keydown', function(e){
            if(e.key === 'ArrowLeft'){ e.preventDefault(); setActive(current - 1); }
            if(e.key === 'ArrowRight'){ e.preventDefault(); setActive(current + 1); }
        });

        setActive(current);
    }

    document.addEventListener('DOMContentLoaded', function(){
        document.querySelectorAll('[data-rg-gallery]').forEach(initGallery);
    });
})();
