<?php

/**
 * Realgest API Class
 * 
 * Handles all API interactions with Realgest
 */
class Realgest_API {

    /**
     * SOAP Client instance
     */
    private $client = null;

    /**
     * API Key
     */
    private $api_key = '';

    /**
     * Agency ID
     */
    private $agency_id = 0;

    /**
     * External data cache
     */
    private $external_data = array();

    /**
     * Constructor
     */
    public function __construct() {
        // Get API settings from options
        $this->api_key = get_option('realgest_api_key', '');
        $this->agency_id = get_option('realgest_agency_id', 0);

        // Initialize SOAP client if API key is set
        if (!empty($this->api_key)) {
            $this->init_client();
        }
    }

    /**
     * Initialize SOAP client
     */
    private function init_client() {
        try {
            $wsdl_path = REALGEST_PLUGIN_DIR . 'resources/realgest-api.wsdl';

            // Check if WSDL file exists
            if (file_exists($wsdl_path)) {
                $this->client = new SoapClient($wsdl_path, array(
                    'trace' => 1,
                    'exceptions' => 1,
                    'cache_wsdl' => WSDL_CACHE_NONE
                ));
            } else {
                error_log('Realgest API WSDL file not found: ' . $wsdl_path);
            }
        } catch (Exception $e) {
            error_log('Realgest API Error: ' . $e->getMessage());
        }
    }

    /**
     * Check if API is configured
     */
    public function is_configured() {
        return (!empty($this->api_key) && !empty($this->agency_id) && $this->client !== null);
    }

    /**
     * Get property listings
     * 
     * @param array $args Search parameters
     * @param int $page Page number
     * @param int $per_page Items per page
     * @return array Properties and pagination info
     */
    public function get_properties($args = array(), $page = 1, $per_page = 10) {
        if (!$this->is_configured()) {
            return array(
                'success' => false,
                'error' => 'API not configured',
                'properties' => array(),
                'total' => 0,
                'pages' => 0
            );
        }

        try {
            // Default parameters
            $params = array(
                'admin_api_key' => $this->api_key,
                'id_agenzia' => $this->agency_id,
                'start' => ($page - 1) * $per_page,
                'limit' => $per_page,
                'ordinamento' => 'data_inserimento',
                'ordinamento_verso' => 'DESC',
                'attivo' => true
            );
			

            // Merge with user parameters
            $params = array_merge($params, $this->prepare_search_params($args));
	        

            // Get total count first
            $total_params = array_merge($params);
            unset($total_params['start']);
            unset($total_params['limit']);
            unset($total_params['ordinamento']);
            unset($total_params['ordinamento_verso']);




	        //$total_result = $this->client->RG_GetTotAnnunci($total_params);
            // Get complete parameter array for the SOAP request
            $complete_params = $this->get_soap_request_parameters('WsAction.RG_GetTotAnnunciRequest');
            // Merge with actual parameter values
            $complete_params = array_merge($complete_params, $total_params);
	        $total_result = $this->client->__soapCall('WsAction.RG_GetTotAnnunci', $complete_params);


            if (!$total_result->Success) {
                return array(
                    'success' => false,
                    'error' => $total_result->Error,
                    'properties' => array(),
                    'total' => 0,
                    'pages' => 0
                );
            }

            $total = $total_result->NumeroAnnunci;
            $pages = ceil($total / $per_page);




	        // Get properties
            // Get complete parameter array for the SOAP request
            $complete_params = $this->get_soap_request_parameters('WsAction.RG_GetAnnunciRequest');



	        // Merge with actual parameter values
            $complete_params = array_merge($complete_params, $params);

	        $result = $this->client->__soapCall('WsAction.RG_GetAnnunci', $complete_params);

	        //mail('claudio@itala.it', 'params2', $this->client->__getLastRequest());


            if (!$result->Success) {
                return array(
                    'success' => false,
                    'error' => $result->Error,
                    'properties' => array(),
                    'total' => 0,
                    'pages' => 0
                );
            }

            // Process properties
            $properties = array();



	        if ($result->NumeroAnnunci > 0 && isset($result->Annunci) && is_array($result->Annunci)) {
                foreach ($result->Annunci as $property) {
                    $properties[] = $this->format_property($property);
                }
            }

            return array(
                'success' => true,
                'error' => '',
                'properties' => $properties,
                'total' => $total,
                'pages' => $pages
            );

        } catch (Exception $e) {
            error_log('Realgest API Error: ' . $e->getMessage());

            return array(
                'success' => false,
                'error' => $e->getMessage(),
                'properties' => array(),
                'total' => 0,
                'pages' => 0
            );
        }
    }

    /**
     * Get single property details
     * 
     * @param int $id Property ID
     * @return array Property details
     */
    public function get_property($id) {
        if (!$this->is_configured()) {
            return array(
                'success' => false,
                'error' => 'API not configured',
                'property' => null
            );
        }

        try {
            $params = array(
                'admin_api_key' => $this->api_key,
                'id' => $id,
                'id_agenzia' => $this->agency_id
            );

            // Get complete parameter array for the SOAP request
            $complete_params = $this->get_soap_request_parameters('WsAction.RG_GetAnnuncioRequest');
            // Merge with actual parameter values
            $complete_params = array_merge($complete_params, $params);
            $result = $this->client->__soapCall('WsAction.RG_GetAnnuncio', $complete_params);

            if (!$result->Success) {
                return array(
                    'success' => false,
                    'error' => $result->Error,
                    'property' => null
                );
            }

            return array(
                'success' => true,
                'error' => '',
                'property' => $this->format_property($result->Annuncio)
            );

        } catch (Exception $e) {
            error_log('Realgest API Error: ' . $e->getMessage());

            return array(
                'success' => false,
                'error' => $e->getMessage(),
                'property' => null
            );
        }
    }

    /**
     * Get property by URL slug
     * 
     * @param string $slug URL slug
     * @return array Property details
     */
    public function get_property_by_slug($slug) {
        if (!$this->is_configured()) {
            return array(
                'success' => false,
                'error' => 'API not configured',
                'property' => null
            );
        }

        try {
            // Get all properties and find the one with matching slug
            $params = array(
                'admin_api_key' => $this->api_key,
                'id_agenzia' => $this->agency_id,
                'start' => 0,
                'limit' => 1000, // Limit to a reasonable number
                'attivo' => true
            );

            // Get complete parameter array for the SOAP request
            $complete_params = $this->get_soap_request_parameters('WsAction.RG_GetAnnunciRequest');
            // Merge with actual parameter values
            $complete_params = array_merge($complete_params, $params);
            $result = $this->client->__soapCall('WsAction.RG_GetAnnunci', $complete_params);

            if (!$result->Success) {
                return array(
                    'success' => false,
                    'error' => $result->Error,
                    'property' => null
                );
            }

            if ($result->NumeroAnnunci > 0 && isset($result->Annunci) && is_array($result->Annunci)) {
                foreach ($result->Annunci as $property) {
                    if ($property->url_rewrite == $slug) {
                        return array(
                            'success' => true,
                            'error' => '',
                            'property' => $this->format_property($property)
                        );
                    }
                }
            }

            return array(
                'success' => false,
                'error' => 'Property not found',
                'property' => null
            );

        } catch (Exception $e) {
            error_log('Realgest API Error: ' . $e->getMessage());

            return array(
                'success' => false,
                'error' => $e->getMessage(),
                'property' => null
            );
        }
    }

    /**
     * Get property types
     * 
     * @return array Property types
     */
    public function get_property_types() {
        // Try to get data from external CSV file first
        $csv_result = $this->get_property_types_from_csv();

        if ($csv_result['success'] && !empty($csv_result['types'])) {
            return $csv_result;
        }

        // Fall back to API if CSV data is not available
        if (!$this->is_configured()) {
            return array(
                'success' => false,
                'error' => 'API not configured',
                'types' => array()
            );
        }

        try {
            $params = array(
                'admin_api_key' => $this->api_key
            );

            // Get complete parameter array for the SOAP request
            $complete_params = $this->get_soap_request_parameters('WsAction.RG_GetTipologieImmobiliRequest');
            // Merge with actual parameter values
            $complete_params = array_merge($complete_params, $params);
            $result = $this->client->__soapCall('WsAction.RG_GetTipologieImmobili', $complete_params);

            if (!$result->Success) {
                return array(
                    'success' => false,
                    'error' => $result->Error,
                    'types' => array()
                );
            }

            $types = array();

            if (isset($result->Elenco) && is_array($result->Elenco)) {
                foreach ($result->Elenco as $type) {
                    $types[] = array(
                        'id' => $type->id,
                        'name' => $type->nome,
                        'class' => $type->classe
                    );
                }
            }

            return array(
                'success' => true,
                'error' => '',
                'types' => $types
            );

        } catch (Exception $e) {
            error_log('Realgest API Error: ' . $e->getMessage());

            return array(
                'success' => false,
                'error' => $e->getMessage(),
                'types' => array()
            );
        }
    }

    /**
     * Get offer types (sale/rent)
     * 
     * @return array Offer types
     */
    public function get_offer_types() {
        // Try to get data from external CSV file first
        $csv_result = $this->get_offer_types_from_csv();

        if ($csv_result['success'] && !empty($csv_result['types'])) {
            return $csv_result;
        }

        // Fall back to API if CSV data is not available
        if (!$this->is_configured()) {
            return array(
                'success' => false,
                'error' => 'API not configured',
                'types' => array()
            );
        }

        try {
            $params = array(
                'admin_api_key' => $this->api_key,
                'tipo' => 'offerta'
            );

            // Get complete parameter array for the SOAP request
            $complete_params = $this->get_soap_request_parameters('WsAction.RG_GetTipologieRequest');
            // Merge with actual parameter values
            $complete_params = array_merge($complete_params, $params);
            $result = $this->client->__soapCall('WsAction.RG_GetTipologie', $complete_params);

            if (!$result->Success) {
                return array(
                    'success' => false,
                    'error' => $result->Error,
                    'types' => array()
                );
            }

            $types = array();

            if (isset($result->Elenco) && is_array($result->Elenco)) {
                foreach ($result->Elenco as $type) {
                    $types[] = array(
                        'id' => $type->id,
                        'name' => $type->nome
                    );
                }
            }

            return array(
                'success' => true,
                'error' => '',
                'types' => $types
            );

        } catch (Exception $e) {
            error_log('Realgest API Error: ' . $e->getMessage());

            return array(
                'success' => false,
                'error' => $e->getMessage(),
                'types' => array()
            );
        }
    }

    /**
     * Get cities
     * 
     * @param string $term Search term
     * @return array Cities
     */
    public function get_cities($term = '') {
        // Try to get data from external CSV file first
        $csv_result = $this->get_cities_from_csv($term);

        if ($csv_result['success'] && !empty($csv_result['cities'])) {
            return $csv_result;
        }

        // Fall back to API if CSV data is not available
        if (!$this->is_configured()) {
            return array(
                'success' => false,
                'error' => 'API not configured',
                'cities' => array()
            );
        }

        try {
            $params = array(
                'admin_api_key' => $this->api_key,
                'term' => $term
            );

            // Get complete parameter array for the SOAP request
            $complete_params = $this->get_soap_request_parameters('WsAction.RG_GetComuneRequest');
            // Merge with actual parameter values
            $complete_params = array_merge($complete_params, $params);
            $result = $this->client->__soapCall('WsAction.RG_GetComune', $complete_params);

            if (!$result->Success) {
                return array(
                    'success' => false,
                    'error' => $result->Error,
                    'cities' => array()
                );
            }

            $cities = array();

            if (isset($result->Elenco) && is_array($result->Elenco)) {
                foreach ($result->Elenco as $city) {
                    $cities[] = array(
                        'id' => $city->id,
                        'name' => $city->nome,
                        'province' => $city->provincia,
                        'province_code' => $city->sigla_provincia
                    );
                }
            }

            return array(
                'success' => true,
                'error' => '',
                'cities' => $cities
            );

        } catch (Exception $e) {
            error_log('Realgest API Error: ' . $e->getMessage());

            return array(
                'success' => false,
                'error' => $e->getMessage(),
                'cities' => array()
            );
        }
    }

    /**
     * Get zones
     * 
     * @param int $city_id City ID
     * @param string $term Search term
     * @return array Zones
     */
    public function get_zones($city_id, $term = '') {
        // Try to get data from external CSV file first
        $csv_result = $this->get_zones_from_csv($city_id, $term);

        if ($csv_result['success'] && !empty($csv_result['zones'])) {
            return $csv_result;
        }

        // Fall back to API if CSV data is not available
        if (!$this->is_configured()) {
            return array(
                'success' => false,
                'error' => 'API not configured',
                'zones' => array()
            );
        }

        try {
            $params = array(
                'admin_api_key' => $this->api_key,
                'idcomune' => $city_id,
                'term' => $term,
                'selected' => ''
            );

            // Get complete parameter array for the SOAP request
            $complete_params = $this->get_soap_request_parameters('WsAction.RG_GetZoneRequest');
            // Merge with actual parameter values
            $complete_params = array_merge($complete_params, $params);
            $result = $this->client->__soapCall('WsAction.RG_GetZone', $complete_params);

            if (!$result->Success) {
                return array(
                    'success' => false,
                    'error' => $result->Error,
                    'zones' => array()
                );
            }

            $zones = array();

            if (isset($result->Zone) && is_array($result->Zone)) {
                foreach ($result->Zone as $zone) {
                    $zones[] = array(
                        'id' => $zone->id_area,
                        'name' => $zone->nome_area,
                        'city' => $zone->nome_comune,
                        'province' => $zone->nome_provincia
                    );
                }
            }

            return array(
                'success' => true,
                'error' => '',
                'zones' => $zones
            );

        } catch (Exception $e) {
            error_log('Realgest API Error: ' . $e->getMessage());

            return array(
                'success' => false,
                'error' => $e->getMessage(),
                'zones' => array()
            );
        }
    }

    /**
     * Format property data
     * 
     * @param object $property Property data from API
     * @return array Formatted property data
     */
	private function format_property($property) {
		$images = array();

		if (isset($property->immagini) && is_array($property->immagini)) {
			foreach ($property->immagini as $image) {
				$images[] = array(
					'url' => $image->path,
					'title' => $image->title,
					'is_main' => $image->principale
				);
			}
		}

		return array(
			'id' => $property->id,
			'reference' => $property->riferimento_annuncio,
			'title' => isset($property->titolo) ? $property->titolo : $property->tipologia . ' ' . $property->offerta,
            'tipologia' => $property->tipologia,
			'slug' => isset($property->url_rewrite) ? $property->url_rewrite : '',
			'type' => $property->tipologia,
			'offer' => $property->offerta,
			'class' => $property->classe,
			'price' => isset($property->prezzo) ? $property->prezzo : '',
			'show_price' => $property->visualizza_prezzo,
			'surface' => $property->superficie,
			'rooms' => isset($property->locali) ? $property->locali : null,
			'bathrooms' => isset($property->servizi) ? $property->servizi : null,
			'description' => $property->descrizione,
			'extended_description' => isset($property->descrizione_estesa) ? $property->descrizione_estesa : '',
			'energy_class' => isset($property->classe_energetica) ? $property->classe_energetica : '',
			'ipe' => isset($property->ipe) ? $property->ipe : '',
			'location' => array(
				'address' => isset($property->indirizzo) ? $property->indirizzo : '',
				'number' => isset($property->civico) ? $property->civico : '',
				'zip' => isset($property->cap) ? $property->cap : '',
				'city' => $property->nome_comune,
				'province' => $property->nome_provincia,
				'zone' => isset($property->nome_area) ? $property->nome_area : '',
				'latitude' => $property->lat,
				'longitude' => $property->long,
				'show_map' => $property->mostra_ubicazione
			),
			'features' => array(
				'garden' => isset($property->giardino) ? $property->giardino : false,
				'terrace' => isset($property->terrazzo) ? $property->terrazzo : false,
				'balcony' => isset($property->balcone) ? $property->balcone : false,
				'elevator' => isset($property->ascensore) ? $property->ascensore : false,
				'cellar' => isset($property->cantina) ? $property->cantina : false,
				'air_conditioning' => isset($property->aria_condizionata) ? $property->aria_condizionata : false,
				'furnished' => isset($property->arredato) ? $property->arredato : false
			),
			'images' => $images,
			'videos' => array(
				isset($property->url_video1) && !empty($property->url_video1) ? $property->url_video1 : null,
				isset($property->url_video2) && !empty($property->url_video2) ? $property->url_video2 : null,
				isset($property->url_video3) && !empty($property->url_video3) ? $property->url_video3 : null,
				isset($property->url_video4) && !empty($property->url_video4) ? $property->url_video4 : null,
				isset($property->url_video5) && !empty($property->url_video5) ? $property->url_video5 : null,
				isset($property->url_video6) && !empty($property->url_video6) ? $property->url_video6 : null
			),
			'virtual_tour' => isset($property->video_360) ? $property->video_360 : '',
			'floor_plan' => isset($property->planimetria) ? $property->planimetria : '',
			'date' => $property->data_inserimento
		);
	}

    /**
     * Read CSV file
     * 
     * @param string $file_name File name (without path)
     * @return array CSV data as associative array
     */
    private function read_csv_file($file_name) {
        // Check if data is already cached
        if (isset($this->external_data[$file_name])) {
            return $this->external_data[$file_name];
        }

        $file_path = REALGEST_PLUGIN_DIR . 'resources/dati_esterni/' . $file_name;

        if (!file_exists($file_path)) {
            error_log('Realgest API Error: CSV file not found: ' . $file_path);
            return array();
        }

        $data = array();

        if (($handle = fopen($file_path, "r")) !== FALSE) {
            // Get headers
            $headers = fgetcsv($handle, 1000, ",");

            // Remove quotes from headers
            foreach ($headers as &$header) {
                $header = trim($header, '"');
            }

            // Read data
            while (($row = fgetcsv($handle, 1000, ",")) !== FALSE) {
                $item = array();

                foreach ($row as $key => $value) {
                    if (isset($headers[$key])) {
                        $item[$headers[$key]] = trim($value, '"');
                    }
                }

                $data[] = $item;
            }

            fclose($handle);
        }

        // Cache data
        $this->external_data[$file_name] = $data;

        return $data;
    }

    /**
     * Get property types from external data
     * 
     * @return array Property types
     */
    public function get_property_types_from_csv() {
        $data = $this->read_csv_file('id_tipologia_immobile.csv');

        $types = array();

        foreach ($data as $item) {
            $types[] = array(
                'id' => $item['id_tipologia'],
                'name' => isset($item['nome']) ? $item['nome'] : '',
                'class' => $item['id_classe_immobile']
            );
        }

        return array(
            'success' => true,
            'error' => '',
            'types' => $types
        );
    }

    /**
     * Get property classes from external data
     * 
     * @return array Property classes
     */
    public function get_property_classes_from_csv() {
        $data = $this->read_csv_file('id_classe_immobile.csv');

        $classes = array();

        foreach ($data as $item) {
            $classes[] = array(
                'id' => $item['id_classe_immobile'],
                'name' => isset($item['nome']) ? $item['nome'] : ''
            );
        }

        return array(
            'success' => true,
            'error' => '',
            'classes' => $classes
        );
    }

    /**
     * Get offer types from external data
     * 
     * @return array Offer types
     */
    public function get_offer_types_from_csv() {
        $data = $this->read_csv_file('id_tipologia_offerta.csv');

        $types = array();

        foreach ($data as $item) {
            $types[] = array(
                'id' => $item['id_tipologia_offerta'],
                'name' => isset($item['nome']) ? $item['nome'] : ''
            );
        }

        return array(
            'success' => true,
            'error' => '',
            'types' => $types
        );
    }

    /**
     * Get cities from external data
     * 
     * @param string $term Search term
     * @return array Cities
     */
    public function get_cities_from_csv($term = '') {
        $data = $this->read_csv_file('id_comune.csv');


        $cities = array();

        foreach ($data as $item) {
            // Filter by term if provided
            if (!empty($term) && (!isset($item['nome_comune']) || stripos($item['nome_comune'], $term) === false)) {
                continue;
            }

            $cities[] = array(
                'id' => $item['id_comune'],
                'name' => isset($item['nome_comune']) ? $item['nome_comune'] : '',
                'province' => isset($item['id_provincia']) ? $this->get_province_name($item['id_provincia']) : '',
                'province_code' => isset($item['id_provincia']) ? $this->get_province_code($item['id_provincia']) : ''
            );
        }

        return array(
            'success' => true,
            'error' => '',
            'cities' => $cities
        );
    }

    /**
     * Get province name by ID
     * 
     * @param string $province_id Province ID
     * @return string Province name
     */
    private function get_province_name($province_id) {
        $data = $this->read_csv_file('id_provincia.csv');

        foreach ($data as $item) {
            if ($item['id_provincia'] == $province_id) {
                if(isset($item['nome_provincia']) ?? $item['nome_provincia']){
                    return $item['nome_provincia'];
                }
                return '';
            }
        }

        return '';
    }

    /**
     * Get province code by ID
     * 
     * @param string $province_id Province ID
     * @return string Province code
     */
    private function get_province_code($province_id) {
        $data = $this->read_csv_file('id_provincia.csv');

        foreach ($data as $item) {
            if ($item['id_provincia'] == $province_id) {
                return isset($item['sigla_provincia']) ? $item['sigla_provincia'] : '';
            }
        }

        return '';
    }

    /**
     * Get zones from external data
     * 
     * @param string $city_id City ID
     * @param string $term Search term
     * @return array Zones
     */
    public function get_zones_from_csv($city_id, $term = '') {
        $data = $this->read_csv_file('id_zona.csv');

        $zones = array();

        foreach ($data as $item) {
            // Filter by city ID
            if (!empty($city_id) && $item['id_comune'] != $city_id) {
                continue;
            }

            // Filter by term if provided
            if (!empty($term) && (!isset($item['nome_zona']) || stripos($item['nome_zona'], $term) === false)) {
                continue;
            }

            $nome_zona = '';
            if(isset($item['nome_zona']) ?? $item['nome_zona']){
              $nome_zona = $item['nome_zona'];
            }

            $nome_provincia = '';
            if(isset($item['id_provincia'])){
                $nome_provincia = $this->get_province_name($item['id_provincia']);
            }

            $zones[] = array(
                'id' => $item['id_zona'],
                'name' => $nome_zona,
                'city' => $this->get_city_name($item['id_comune']),
                'province' => $nome_provincia
            );
        }

        return array(
            'success' => true,
            'error' => '',
            'zones' => $zones
        );
    }

    /**
     * Get city name by ID
     * 
     * @param string $city_id City ID
     * @return string City name
     */
    private function get_city_name($city_id) {
        $data = $this->read_csv_file('id_comune.csv');

        foreach ($data as $item) {
            if ($item['id_comune'] == $city_id) {
                return isset($item['nome_comune']) ? $item['nome_comune'] : '';
            }
        }

        return '';
    }

    /**
     * Prepare search parameters
     * 
     * @param array $args Search arguments
     * @return array Formatted search parameters
     */
    private function prepare_search_params($args) {
        $params = array();

        // Property type
        if (isset($args['type']) && !empty($args['type'])) {
            $params['id_tipologia'] = intval($args['type']);
        }

        // Offer type (sale/rent)
        if (isset($args['offer']) && !empty($args['offer'])) {
            $params['id_tipo_offerta'] = intval($args['offer']);
        }

        // Property class (residential, commercial, etc.)
        if (isset($args['class']) && !empty($args['class'])) {
            $params['id_classe'] = intval($args['class']);
        }

        // City
        if (isset($args['city']) && !empty($args['city'])) {
            $params['id_comune'] = intval($args['city']);
        }

        // Zone
        if (isset($args['zone']) && !empty($args['zone'])) {
            $params['id_area'] = intval($args['zone']);
        }

        // Price range
        if (isset($args['price_min']) && !empty($args['price_min'])) {
            $params['prezzo_min'] = floatval($args['price_min']);
        }
        if (isset($args['price_max']) && !empty($args['price_max'])) {
            $params['prezzo_max'] = floatval($args['price_max']);
        }

        // Surface range
        if (isset($args['surface_min']) && !empty($args['surface_min'])) {
            $params['superficie_min'] = intval($args['surface_min']);
        }
        if (isset($args['surface_max']) && !empty($args['surface_max'])) {
            $params['superficie_max'] = intval($args['surface_max']);
        }

        // Rooms range
        if (isset($args['rooms_min']) && !empty($args['rooms_min'])) {
            $params['locali_min'] = intval($args['rooms_min']);
        }
        if (isset($args['rooms_max']) && !empty($args['rooms_max'])) {
            $params['locali_max'] = intval($args['rooms_max']);
        }

        // Bathrooms range
        if (isset($args['bathrooms_min']) && !empty($args['bathrooms_min'])) {
            $params['servizi_min'] = intval($args['bathrooms_min']);
        }
        if (isset($args['bathrooms_max']) && !empty($args['bathrooms_max'])) {
            $params['servizi_max'] = intval($args['bathrooms_max']);
        }

        // Features
        if (isset($args['garden']) && $args['garden']) {
            $params['giardino'] = true;
        }
        if (isset($args['terrace']) && $args['terrace']) {
            $params['terrazzo'] = true;
        }
        if (isset($args['balcony']) && $args['balcony']) {
            $params['balcone'] = true;
        }
        if (isset($args['elevator']) && $args['elevator']) {
            $params['ascensore'] = true;
        }
        if (isset($args['cellar']) && $args['cellar']) {
            $params['cantina'] = true;
        }
        if (isset($args['air_conditioning']) && $args['air_conditioning']) {
            $params['aria_condizionata'] = true;
        }
        if (isset($args['furnished']) && $args['furnished']) {
            $params['arredato'] = true;
        }

        // Only with photos
        if (isset($args['only_with_photos']) && $args['only_with_photos']) {
            $params['solo_con_foto'] = true;
        }

        // Sorting
        if (isset($args['orderby']) && !empty($args['orderby'])) {
            $params['ordinamento'] = $args['orderby'];
        }
        if (isset($args['order']) && !empty($args['order'])) {
            $params['ordinamento_verso'] = $args['order'];
        }

        return $params;
    }

    /**
     * Get SOAP request parameters
     * 
     * @param string $request_name Name of the SOAP request (e.g., "WsAction.RG_GetAnnunciRequest")
     * @return array Array with parameter names as keys and NULL as values
     */
    public function get_soap_request_parameters($request_name) {
        $params = array();

        // Path to the WSDL file
        $wsdl_path = REALGEST_PLUGIN_DIR . 'resources/realgest-api.wsdl';

        // Check if WSDL file exists
        if (!file_exists($wsdl_path)) {
            return $params;
        }

        // Load and parse the WSDL file
        $wsdl_content = file_get_contents($wsdl_path);

        // Create a SimpleXML object from the WSDL content
        $xml = simplexml_load_string($wsdl_content);

        // Register namespaces
        $xml->registerXPathNamespace('wsdl', 'http://schemas.xmlsoap.org/wsdl/');
        $xml->registerXPathNamespace('xsd', 'http://www.w3.org/2001/XMLSchema');

        // Find the message with the given name
        $message = $xml->xpath("//wsdl:message[@name='$request_name']");

        // If message found, extract its parts (parameters)
        if (!empty($message)) {
            $parts = $message[0]->xpath('./wsdl:part');

            // For each part, add its name as a key with NULL value
            foreach ($parts as $part) {
                $name = (string)$part['name'];
                $params[$name] = NULL;
            }
        }

        return $params;
    }
}
