<?php

/**
 * Realgest Shortcodes Class
 * 
 * Handles all shortcodes for displaying Realgest properties
 */
class Realgest_Shortcodes {

    /**
     * API instance
     */
    private $api = null;

    /**
     * Constructor
     */
    public function __construct($api) {
        $this->api = $api;

        // Register shortcodes
        add_shortcode('realgest_properties', array($this, 'properties_shortcode'));
        add_shortcode('realgest_property', array($this, 'property_shortcode'));
        add_shortcode('realgest_search', array($this, 'search_shortcode'));
    }

    public function register_styles($template) {
        wp_register_style(
                'realgest-styles',
                $template,
                array(),
                REALGEST_PLUGIN_VERSION
        );
        wp_enqueue_style('realgest-styles');
    }

    /**
     * Properties shortcode
     * 
     * @param array $atts Shortcode attributes
     * @return string Shortcode output
     */
    public function properties_shortcode($atts) {
        // Extract attributes
        $atts = shortcode_atts(array(
            'limit' => 10,
            'type' => '',
            'offer' => '',
            'class' => '',
            'city' => '',
            'zone' => '',
            'price_min' => '',
            'price_max' => '',
            'surface_min' => '',
            'surface_max' => '',
            'rooms_min' => '',
            'rooms_max' => '',
            'bathrooms_min' => '',
            'bathrooms_max' => '',
            'garden' => false,
            'terrace' => false,
            'balcony' => false,
            'elevator' => false,
            'cellar' => false,
            'air_conditioning' => false,
            'furnished' => false,
            'only_with_photos' => true,
            'orderby' => 'data_inserimento',
            'order' => 'DESC',
            'template' => 'grid',
            'columns' => 3,
            'pagination' => true
        ), $atts);

        // Merge with URL parameters for filtering
        $url_params = array(
            'type' => isset($_GET['type']) ? sanitize_text_field($_GET['type']) : '',
            'offer' => isset($_GET['offer']) ? sanitize_text_field($_GET['offer']) : '',
            'class' => isset($_GET['class']) ? sanitize_text_field($_GET['class']) : '',
            'city' => isset($_GET['city']) ? sanitize_text_field($_GET['city']) : '',
            'zone' => isset($_GET['zone']) ? sanitize_text_field($_GET['zone']) : '',
            'price_min' => isset($_GET['price_min']) ? sanitize_text_field($_GET['price_min']) : '',
            'price_max' => isset($_GET['price_max']) ? sanitize_text_field($_GET['price_max']) : '',
            'surface_min' => isset($_GET['surface_min']) ? sanitize_text_field($_GET['surface_min']) : '',
            'surface_max' => isset($_GET['surface_max']) ? sanitize_text_field($_GET['surface_max']) : '',
            'rooms_min' => isset($_GET['rooms_min']) ? sanitize_text_field($_GET['rooms_min']) : '',
            'rooms_max' => isset($_GET['rooms_max']) ? sanitize_text_field($_GET['rooms_max']) : '',
            'bathrooms_min' => isset($_GET['bathrooms_min']) ? sanitize_text_field($_GET['bathrooms_min']) : '',
            'bathrooms_max' => isset($_GET['bathrooms_max']) ? sanitize_text_field($_GET['bathrooms_max']) : '',
            'garden' => isset($_GET['garden']) ? sanitize_text_field($_GET['garden']) : false,
            'terrace' => isset($_GET['terrace']) ? sanitize_text_field($_GET['terrace']) : false,
            'balcony' => isset($_GET['balcony']) ? sanitize_text_field($_GET['balcony']) : false,
            'elevator' => isset($_GET['elevator']) ? sanitize_text_field($_GET['elevator']) : false,
            'cellar' => isset($_GET['cellar']) ? sanitize_text_field($_GET['cellar']) : false,
            'air_conditioning' => isset($_GET['air_conditioning']) ? sanitize_text_field($_GET['air_conditioning']) : false,
            'furnished' => isset($_GET['furnished']) ? sanitize_text_field($_GET['furnished']) : false,
        );


        // Override shortcode attributes with URL parameters if they exist and are not empty
        foreach ($url_params as $param => $value) {
            if (!empty($value)) {
                $atts[$param] = $value;
            }
        }

        // Convert string boolean values to actual booleans
        foreach (array('garden', 'terrace', 'balcony', 'elevator', 'cellar', 'air_conditioning', 'furnished', 'only_with_photos', 'pagination') as $bool_attr) {
            if (isset($atts[$bool_attr])) {
                $atts[$bool_attr] = filter_var($atts[$bool_attr], FILTER_VALIDATE_BOOLEAN);
            }
        }

        // Convert numeric values
        foreach (array('limit', 'type', 'offer', 'class', 'city', 'zone', 'price_min', 'price_max', 'surface_min', 'surface_max', 'rooms_min', 'rooms_max', 'bathrooms_min', 'bathrooms_max', 'columns') as $num_attr) {
            if (isset($atts[$num_attr]) && $atts[$num_attr] !== '') {
                $atts[$num_attr] = is_numeric($atts[$num_attr]) ? $atts[$num_attr] : '';
            }
        }
		

        // Get current page
        $page = (int) ( get_query_var('paged') ?: get_query_var('page') ?: ( isset($_GET['paged']) ? $_GET['paged'] : 1 ) );
        if ($page < 1) { $page = 1; }

        // Get properties
        $result = $this->api->get_properties($atts, $page, $atts['limit']);

        // Start output buffering
        ob_start();

        // Check if API is configured
        if (!$this->api->is_configured()) {
            echo '<div class="realgest-error">' . __('API Realgest non configurata. Configura l\'API nelle impostazioni del plugin.', 'realgest-for-wordpress') . '</div>';
            return ob_get_clean();
        }

        // Check for errors
        if (!$result['success']) {
            echo '<div class="realgest-error">' . $result['error'] . '</div>';
            return ob_get_clean();
        }

        // Check if properties exist
        if (empty($result['properties'])) {
            echo '<div class="realgest-no-results">' . __('Nessun immobile trovato.', 'realgest-for-wordpress') . '</div>';
            return ob_get_clean();
        }

        // === Layout/CSS via Registry ===
if (!class_exists('Realgest_Style_Registry')) {
    // Fallback safety: include if not loaded
    if (defined('REALGEST_PLUGIN_DIR') && file_exists(REALGEST_PLUGIN_DIR . 'includes/class-realgest-style-registry.php')) {
        require_once REALGEST_PLUGIN_DIR . 'includes/class-realgest-style-registry.php';
    }
}
$registry = class_exists('Realgest_Style_Registry') ? Realgest_Style_Registry::get_registry() : [];
$layout   = !empty($atts['template']) ? sanitize_key($atts['template']) : 'grid'; // 'grid' | 'vertical'
if (!isset($registry['archive'][$layout])) {
    $layout = 'grid';
}
$tpl_rel = $registry['archive'][$layout]['template'] ?? 'templates/properties/properties-grid.php';
$css_rel = $registry['archive'][$layout]['css']      ?? 'assets/css/properties/properties-grid.css';

// Resolve template absolute path (override via templates loader if you neeed it)
$template_path_php = trailingslashit(REALGEST_PLUGIN_DIR) . ltrim($tpl_rel, '/');
if (!file_exists($template_path_php)) {
    $template_path_php = REALGEST_PLUGIN_DIR . 'templates/properties/properties-grid.php';
}

// Enqueue CSS with theme override support
if (!function_exists('realgest_enqueue_asset_css')) {
    if (defined('REALGEST_PLUGIN_DIR') && file_exists(REALGEST_PLUGIN_DIR . 'includes/functions-assets.php')) {
        require_once REALGEST_PLUGIN_DIR . 'includes/functions-assets.php';
    }
}
if (function_exists('realgest_enqueue_asset_css')) {
    realgest_enqueue_asset_css($css_rel, 'realgest-archive-' . $layout);
} else {
    // Legacy enqueue (absolute URL)
    $template_path_css = plugins_url($css_rel, REALGEST_PLUGIN_FILE);
    $this->register_styles($template_path_css);
}

// Include template
include $template_path_php;

// Pagination
if (!empty($atts['pagination']) && !empty($result['pages']) && intval($result['pages']) > 1) {

    // Preserve query args except pagination vars
    $add_args = array();
    if (!empty($_GET) && is_array($_GET)) {
        foreach ($_GET as $k => $v) {
            if ($k === 'paged' || $k === 'page') continue;
            $add_args[$k] = is_array($v) ? array_map('sanitize_text_field', $v) : sanitize_text_field($v);
        }
    }

    $total_pages = intval($result['pages']);
    $current     = max(1, intval($page));

    // Robust base/format for both pretty and non-pretty permalinks
    $base_link = remove_query_arg(array('paged','page'), get_pagenum_link(1));
    $permalink = get_option('permalink_structure');
    if ($permalink) {
        // pretty permalinks
        $base   = trailingslashit($base_link) . '%_%';
        $format = 'page/%#%/';
    } else {
        // query-string permalinks
        $base   = $base_link . '%_%';
        $format = (strpos($base_link, '?') === false ? '?paged=%#%' : '&paged=%#%');
    }

    echo '<nav class="realgest-pagination" role="navigation" aria-label="Paginazione immobili">';
    echo paginate_links(array(
        'base'      => $base,
        'format'    => $format,
        'current'   => $current,
        'total'     => $total_pages,
        'prev_text' => '&laquo;',
        'next_text' => '&raquo;',
        'add_args'  => $add_args,
        'type'      => 'list'
    ));
    echo '</nav>';
}


        // Return the buffered content
        return ob_get_clean();
    }

    /**
     * Property shortcode
     * 
     * @param array $atts Shortcode attributes
     * @return string Shortcode output
     */
    public function property_shortcode($atts) {
        // Extract attributes
        $atts = shortcode_atts(array(
            'id' => 0,
            'slug' => '',
            'template' => 'default'
        ), $atts);

        // Start output buffering
        ob_start();

        // Check if API is configured
        if (!$this->api->is_configured()) {
            echo '<div class="realgest-error">' . __('API Realgest non configurata. Configura l\'API nelle impostazioni del plugin.', 'realgest-for-wordpress') . '</div>';
            return ob_get_clean();
        }

        // Get property
        $result = null;

        if (!empty($atts['id'])) {
            $result = $this->api->get_property($atts['id']);
        } elseif (!empty($atts['slug'])) {
            $result = $this->api->get_property_by_slug($atts['slug']);
        } else {
            echo '<div class="realgest-error">' . __('ID o slug dell\'immobile richiesto.', 'realgest-for-wordpress') . '</div>';
            return ob_get_clean();
        }

        // Check for errors
        if (!$result['success']) {
            echo '<div class="realgest-error">' . $result['error'] . '</div>';
            return ob_get_clean();
        }

        // Check if property exists
        if (empty($result['property'])) {
            echo '<div class="realgest-no-results">' . __('Immobile non trovato.', 'realgest-for-wordpress') . '</div>';
            return ob_get_clean();
        }

        $property = $result['property'];

        $template_file_slug = 'single-property-' . $atts['template'];

        $template_file_php = $template_file_slug . '.php';
        $template_file_css = $template_file_slug . '.css';

        $template_path_php = REALGEST_PLUGIN_DIR . 'templates/single-property/' . $template_file_php;
        $template_path_css = REALGEST_PLUGIN_URL . 'assets/css/single-property/' . $template_file_css;

        if (!file_exists($template_path_php)) {
            $template_path_php = REALGEST_PLUGIN_DIR . 'templates/single-property/single-property-default.php';
        }
        if (!file_exists($template_path_css)) {
            $template_path_css = REALGEST_PLUGIN_URL . 'assets/css/single-property/single-property-default.css';
        }
        $this->register_styles($template_path_css);
        // Include template
        include $template_path_php;

        // Return the buffered content
        return ob_get_clean();
    }

    /**
     * Search shortcode
     * 
     * @param array $atts Shortcode attributes
     * @return string Shortcode output
     */
    public function search_shortcode($atts) {
        // Extract attributes
        $atts = shortcode_atts(array(
            'template' => 'default',
            'action' => '',
            'show_offer_types' => true,
            'show_property_types' => true,
            'show_cities' => true,
            'show_zones' => true,
            'show_price' => true,
            'show_surface' => true,
            'show_rooms' => true,
            'show_bathrooms' => true,
            'show_features' => true
        ), $atts);

        // Convert string boolean values to actual booleans
        foreach (array('show_offer_types', 'show_property_types', 'show_cities', 'show_zones', 'show_price', 'show_surface', 'show_rooms', 'show_bathrooms', 'show_features') as $bool_attr) {
            if (isset($atts[$bool_attr])) {
                $atts[$bool_attr] = filter_var($atts[$bool_attr], FILTER_VALIDATE_BOOLEAN);
            }
        }

        // Start output buffering
        ob_start();

        // Check if API is configured
        if (!$this->api->is_configured()) {
            echo '<div class="realgest-error">' . __('API Realgest non configurata. Configura l\'API nelle impostazioni del plugin.', 'realgest-for-wordpress') . '</div>';
            return ob_get_clean();
        }

        // Get offer types
        $offer_types = array();
        if ($atts['show_offer_types']) {
            $offer_result = $this->api->get_offer_types();
            if ($offer_result['success']) {
                $offer_types = $offer_result['types'];
            }
        }

        // Get property types
        $property_types = array();
        if ($atts['show_property_types']) {
            $types_result = $this->api->get_property_types();
            if ($types_result['success']) {
                $property_types = $types_result['types'];
            }
        }

        // Get cities
        $cities = array();
        if ($atts['show_cities']) {
            $cities_result = $this->api->get_cities();
            if ($cities_result['success']) {
                $cities = $cities_result['cities'];
            }
        }

        $template_file_slug = 'search-' . $atts['template'];

        $template_file_php = $template_file_slug . '.php';
        $template_file_css = $template_file_slug . '.css';

        $template_path_php = REALGEST_PLUGIN_DIR . 'templates/search/' . $template_file_php;
        $template_path_css = REALGEST_PLUGIN_URL . 'assets/css/search/' . $template_file_css;

        if (!file_exists($template_path_php)) {
            $template_path_php = REALGEST_PLUGIN_DIR . 'templates/search/search-default.php';
        }
        if (!file_exists($template_path_css)) {
            $template_path_css = REALGEST_PLUGIN_URL . 'assets/css/search/search-default.css';
        }
        
        $this->register_styles($template_path_css);
        // Include template
        include $template_path_php;

// Optionally render results inline if the page doesn't include [realgest_properties]
global $post;
$has_properties_shortcode = ( isset($post) && is_object($post) && has_shortcode($post->post_content ?? '', 'realgest_properties') );

// Detect if there are search query params present
$query_keys = array('type','offer','class','city','zone','price_min','price_max','surface_min','surface_max','rooms_min','rooms_max','bathrooms_min','bathrooms_max','garden','terrace','balcony','elevator','cellar','air_conditioning','furnished');
$has_query = false;
foreach ($query_keys as $qk) {
    if (isset($_GET[$qk]) && $_GET[$qk] !== '' && $_GET[$qk] !== '0') { $has_query = true; break; }
}

if (!$has_properties_shortcode && $has_query) {
    // Build an attributes string forwarding GET params
    $forward = array();
    foreach ($query_keys as $qk) {
        if (isset($_GET[$qk]) && $_GET[$qk] !== '') {
            $val = is_array($_GET[$qk]) ? reset($_GET[$qk]) : $_GET[$qk];
            $forward[] = $qk . '="' . esc_attr($val) . '"';
        }
    }
    // Default to grid layout with pagination
    $forward[] = 'template="grid"';
    $forward[] = 'pagination="true"';
    echo do_shortcode('[realgest_properties ' . implode(' ', $forward) . ']');
}

        // Return the buffered content
        return ob_get_clean();
    }
}