<?php

/**
 * Realgest Templates Class
 * 
 * Handles template loading and overriding
 */
class Realgest_Templates {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Add template directory filter
        add_filter('realgest_template_path', array($this, 'template_path'));
    }
    
    /**
     * Get template path
     * 
     * Checks if a template exists in the theme directory, otherwise uses the plugin template
     * 
     * @param string $template Template file name
     * @return string Template path
     */
    public function template_path($template) {
        // Check if template exists in theme
        $theme_template = locate_template('realgest/' . $template);
        
        if ($theme_template) {
            return $theme_template;
        }
        
        // Otherwise use plugin template
        return REALGEST_PLUGIN_DIR . 'templates/' . $template;
    }
    
    /**
     * Get template
     * 
     * @param string $template_name Template file name
     * @param array $args Template arguments
     * @param string $template_path Template path
     * @param string $default_path Default path
     */
    public static function get_template($template_name, $args = array(), $template_path = '', $default_path = '') {
        if (!empty($args) && is_array($args)) {
            extract($args);
        }
        
        $located = self::locate_template($template_name, $template_path, $default_path);
        
        if (!file_exists($located)) {
            _doing_it_wrong(__FUNCTION__, sprintf('<code>%s</code> does not exist.', $located), '1.0');
            return;
        }
        
        // Allow 3rd party plugin filter template file from their plugin
        $located = apply_filters('realgest_get_template', $located, $template_name, $args, $template_path, $default_path);
        
        do_action('realgest_before_template_part', $template_name, $template_path, $located, $args);
        
        include($located);
        
        do_action('realgest_after_template_part', $template_name, $template_path, $located, $args);
    }
    
    /**
     * Locate template
     * 
     * @param string $template_name Template file name
     * @param string $template_path Template path
     * @param string $default_path Default path
     * @return string Template path
     */
    public static function locate_template($template_name, $template_path = '', $default_path = '') {
        if (!$template_path) {
            $template_path = 'realgest/';
        }
        
        if (!$default_path) {
            $default_path = REALGEST_PLUGIN_DIR . 'templates/';
        }
        
        // Look within passed path within the theme - this is priority
        $template = locate_template(array(
            trailingslashit($template_path) . $template_name,
            $template_name
        ));
        
        // Get default template
        if (!$template) {
            $template = trailingslashit($default_path) . $template_name;
        }
        
        // Return what we found
        return apply_filters('realgest_locate_template', $template, $template_name, $template_path);
    }
}