<?php

/*
Plugin Name: Realgest For Wordpress
Plugin URI: https://github.com/claudio/realgest-for-wordpress
Description: Plugin per integrare gli immobili da Realgest nel tuo sito WordPress.
Version: 1.0.0
Author: RealGest
Author URI: https://www.realgest.it/
License: GPL2
 */

// Define main plugin file
if (!defined('REALGEST_PLUGIN_FILE')) define('REALGEST_PLUGIN_FILE', __FILE__);
/*
*/

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('REALGEST_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('REALGEST_PLUGIN_URL', plugin_dir_url(__FILE__));
define('REALGEST_PLUGIN_VERSION', '1.0.0');

// Include required files
require_once REALGEST_PLUGIN_DIR . 'includes/class-realgest-api.php';
require_once REALGEST_PLUGIN_DIR . 'includes/class-realgest-shortcodes.php';
require_once REALGEST_PLUGIN_DIR . 'includes/class-realgest-templates.php';
require_once REALGEST_PLUGIN_DIR . 'includes/class-realgest-ajax.php';
require_once REALGEST_PLUGIN_DIR . 'admin/class-realgest-admin.php';

/**
 * Main plugin class
 */
class Realgest_For_Wordpress {

    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * API instance
     */
    public $api = null;

    /**
     * Return an instance of this class
     */
    public static function get_instance() {
        if (null == self::$instance) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Initialize the plugin
        add_action('plugins_loaded', array($this, 'init'));

        // Register activation and deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // Register scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'register_scripts'));
    }

    /**
     * Initialize the plugin
     */
    public function init() {
        // Initialize API
        $this->api = new Realgest_API();

        // Initialize shortcodes
        $shortcodes = new Realgest_Shortcodes($this->api);

        // Initialize admin
        if (is_admin()) {
            $admin = new Realgest_Admin($this->api);
        }

        // Initialize templates
        $templates = new Realgest_Templates();

        // Initialize AJAX handler
        $ajax = new Realgest_AJAX($this->api);

        // Add rewrite rules for property detail pages
        add_action('init', array($this, 'add_rewrite_rules'));

        // Add query vars
        add_filter('query_vars', array($this, 'add_query_vars'));

        // Template include filter
        add_filter('template_include', array($this, 'template_include'));
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Create necessary database tables if needed

        // Add rewrite rules
        $this->add_rewrite_rules();

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Register scripts and styles
     */
    public function register_scripts() {
        // Register and enqueue JavaScript
        wp_register_script('realgest-scripts', REALGEST_PLUGIN_URL . 'assets/js/realgest.js', array('jquery'), REALGEST_PLUGIN_VERSION, true);
        wp_enqueue_script('realgest-scripts');

        // Localize script with data
        wp_localize_script('realgest-scripts', 'realgest_params', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'plugin_url' => REALGEST_PLUGIN_URL,
            'nonce' => wp_create_nonce('realgest_nonce'),
            'loading_text' => __('Caricamento...', 'realgest-for-wordpress')
        ));
    }

    /**
     * Add rewrite rules for property detail pages
     */
    public function add_rewrite_rules() {
        add_rewrite_rule(
            'immobili/([^/]+)/?$',
            'index.php?realgest_property=$matches[1]',
            'top'
        );
    }

    /**
     * Add query vars
     */
    public function add_query_vars($vars) {
        $vars[] = 'realgest_property';
        return $vars;
    }

    /**
     * Template include filter
     */
    public function template_include($template) {
        // Check if this is a property detail page
        if (get_query_var('realgest_property')) {
            $property_slug = get_query_var('realgest_property');

            // Load the property detail template
            $new_template = REALGEST_PLUGIN_DIR . 'templates/single-property.php';

            if (file_exists($new_template)) {
                return $new_template;
            }
        }

        return $template;
    }
}

// Initialize the plugin
function realgest_for_wordpress() {
    return Realgest_For_Wordpress::get_instance();
}

require_once REALGEST_PLUGIN_DIR . 'includes/class-realgest-style-registry.php';
require_once REALGEST_PLUGIN_DIR . 'includes/functions-assets.php';

// Start the plugin
realgest_for_wordpress();
